<?php

namespace App\Http\UseCases\Api\Signataire;

use App\Mail\DocumentSignedDownloadNotification;
use App\Models\Signataire;
use App\Models\User;
use App\Notifications\RenouvellementCertificatNotification\EntrepriseNotification;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Notification;
use App\Mail\PendingSignatureNotification;
use App\Mail\AcceptedSignatureNotification;
use App\Mail\DocumentsSignedNotification;
use Illuminate\Support\Facades\Mail;
use App\Http\Services\SignatureService;
use App\Http\UseCases\Api\AbstractUseCase;
use App\Models\Certificat;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\JsonResponse;

final class StoreUseCase extends AbstractUseCase
{
    public function __construct(
        public SignatureService $signatureService,
    ) {}

    private function certificate(User $user) 
    {
        return is_null($user->certificat)
            ? $user->entreprise->certificat
            : $user->certificat;
    }

    public function handle(array $data, User $user)
    {
        $signataires = !isset($data['signataires']) ? [] : json_decode($data['signataires'], true);

        if ($data['action'] === 'faire_signer') {
            $this->storeSignataires($user->id, $data['document'], $signataires, $user->entreprise?->otp_sms);
            return $this->success('Document envoyé avec succès');
        }

        return $this->sendRequest($user, $data, $data['document'], $signataires, false);
    }

    public function sendRequest(User $user, array $data, string $filepath, array $signataires, bool $update): JsonResponse
    {
        $certificat = $this->certificate($user);
        
        if ((int) $certificat->volume_signatures_restant === 0) {
            return $this->error('Le volume de signature épuisé', 403);
        }
        
        if (now()->gte(Carbon::parse($certificat->date_expiration))) {
            return $this->error('Le certificat a expiré', 403);
        }
        
        $response = $this->signatureService->signDocument($certificat, $filepath);
            
        if ($response && $response['error'] === false) {
            if (!$update) {
                $this->store($user, $filepath, $signataires);
            } else {
                $this->update($signataires[0], $filepath);
            }

            $this->updateRemainingSignatureCount($user, $certificat);
            return $this->success($response);
        }

        return $this->error('Une erreur inconnue est survenue lors de la signautre du document', 500);
    }

    private function store(User $user, string $filepath, array $signataires)
    {
        Signataire::factory()->create([
            'user_id' => $user->id,
            'telephone' => $user->telephone,
            'email' => $user->email,
            'nom' => $user->getFullName(),
            'document' => $filepath,
            'statut' => empty($signataires) ? Signataire::STATUT_SIGNE : Signataire::STATUT_EN_COURS,
        ]);

        if (empty($signataires)) { 
            return; 
        }

        $this->storeSignataires($user->id, $filepath, $signataires, $user->entreprise?->otp_sms);
    }

    private function update($signataire, string $filepath)
    {
        Signataire::query()
            ->where('telephone', $signataire['telephone'])
            ->where('document', $filepath)
            ->update([
                'statut' => Signataire::STATUT_SIGNE,
                'updated_at' => now()
            ]);

        $_signataire = Signataire::query()
            ->where('statut', Signataire::STATUT_EN_ATTENTE)
            ->where('document', $filepath)
            ->orderBy('ordre')
            ->first();

        // Si plus aucun signataire, on met le statut à STATUT_SIGNE pour l'utilisateur qui a initié l'action
        if (!$_signataire) {
            $_signataire = Signataire::query()
                ->where('statut', Signataire::STATUT_EN_COURS)
                ->where('document', $filepath)
                ->first();

            $signataires = Signataire::query()
                ->where('document', $filepath)
                ->get();

            if ($_signataire) {
                $_signataire->update(['statut' => Signataire::STATUT_SIGNE]);
            } else {
                $_signataire = Signataire::query()
                    ->where('document', $filepath)
                    ->first();
            }
            
            $this->sendSignedDocumentNotification($_signataire, $signataires);

            return Signataire::query()
                ->where('document', $filepath)
                ->update(['code_auth' => null]);
        }
        
        // On notifie l'initiateur de la signature que le signataire a signé le document
        try {
            Mail::to($_signataire->user->email)
                ->send(
                    new AcceptedSignatureNotification(
                        'https://anrmp.dkbsign.com',
                        $signataire['nom'],
                        File::basename($signataire['document'])
                    )
                );
        } catch (Exception $e) {}

        // On envoie la notification de document en attente de signature au prochain signataire
        try {
            Mail::to($_signataire->email)
                ->send(
                    new PendingSignatureNotification(
                        'https://anrmp.dkbsign.com/signataires?signataire=' . $_signataire->telephone,
                        $_signataire->user->getFullName(),
                        $_signataire->code_auth
                    )
                );
        } catch (Exception $e) {}
    }

    private function sendSignedDocumentNotification(Signataire $signataire, Collection $signataires)
    {
        $document = File::basename($signataire->document);
        $url = $this->signatureService->getSignedDocumentItem($document);
        
        try {
            Mail::to($signataire->user->email)->send(new DocumentsSignedNotification($url, $document));
        } catch (Exception $e) {}

        foreach ($signataires as $signataire) {
            try {
                Mail::to($signataire->email)->send(new DocumentSignedDownloadNotification($url, $document));
            } catch (Exception $e) {}
        }
    }

    private function updateRemainingSignatureCount(User $user, Certificat $certificat): void
    {
        $signatureCount = $this->signatureService->getSignatureCount($certificat);
        $remainingSignaturesCount = $certificat->volume_signatures_total - $signatureCount;

        $certificat->update(['volume_signatures_restant' => $remainingSignaturesCount]);
        $email = !is_null($user->entreprise) ? $user->entreprise->email : $user->email;

        if (in_array($remainingSignaturesCount, [200, 150, 100, 50, 1])) {
            try {
                Notification::route('mail', $email)->notify(new EntrepriseNotification($remainingSignaturesCount));
            } catch (Exception $e) {}
        }
    }

    private function storeSignataires(int $user_id, string $filepath, array $signataires, $otpSMS)
    {
        /*Signataire::query()
                ->where('telephone', '0011223344')
                ->update(['code_auth' => $signataire['code_auth']]);

            Signataire::factory()->create([
                'user_id' => $user_id,
                'document' => $filepath,
                'ordre' => $signataire['ordre'],
                'telephone' => '0011223344',
                'email' => 'eliseekn@gmail.com',
                'nom' => 'John Doe',
                'positionX' => $signataire['position']['x'],
                'positionY' => $signataire['position']['y'],
                'page' => $signataire['page'],
                'code_auth' => $signataire['code_auth']
            ]);

            if ((int) $signataire['ordre'] === 1) {
                try {
                    Mail::to('eliseekn@gmail.com')
                        ->send(
                            new PendingSignatureNotification(
                                'https://anrmp.dkbsign.com/signataires?signataire=0011223344',
                                User::find($user_id)->getFullName(),
                                $signataire['code_auth']
                            )
                        );
                } catch (Exception $e) {}
            }
            
            return;*/
        
        foreach ($signataires as $signataire) {
            /**
             * Vu que le code d'authentification est généré du côté front à chaque requête,
             * on met à jour le nouveau code d'authentification pour tous les documents de ce signataire.
             * De ce fait le signataire pourra toujours signer les anciens documents avec le dernier code
             * d'authentification généré
             */
            Signataire::query()
                ->where('telephone', $signataire['telephone'])
                ->update(['code_auth' => $signataire['code_auth']]);

            Signataire::factory()->create([
                'user_id' => $user_id,
                'document' => $filepath,
                'ordre' => $signataire['ordre'],
                'telephone' => $signataire['telephone'],
                'email' => $signataire['email'],
                'nom' => $signataire['nom'],
                'positionX' => $signataire['position']['x'],
                'positionY' => $signataire['position']['y'],
                'page' => $signataire['page'],
                'code_auth' => $signataire['code_auth']
            ]);

            if ((int) $signataire['ordre'] === 1) {
                try {
                    Mail::to($signataire['email'])
                        ->send(
                            new PendingSignatureNotification(
                                'https://anrmp.dkbsign.com/signataires?signataire=' . $signataire['telephone'],
                                User::find($user_id)->getFullName(),
                                $signataire['code_auth']
                            )
                        );
                } catch (Exception $e) {}
            }
        }
    }
}
