<?php

namespace App\Http\Services;

use App\Models\Certificat;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

final class SignatureService
{
    public $baseUrl = 'https://www.dkbsigns.com/API3/Api_dkbsign3';

    public function url(string $url): string
    {
        return $this->baseUrl . $url;
    }

    protected function sendRequest(string $url, string $method, string $key, array $data = []): array
    {
        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                "authorizations: $key",
                'Content-Type: application/json'
            ],
        ]);

        $response = curl_exec($curl);
        curl_close($curl);
        
        $response = str_replace('﻿', '', trim($response));
        return json_decode($response, true);
    }

    public function signDocument(Certificat $certificat, string $filepath): array
    {
        $data['Key_Api'] = $certificat->cle_api;
        $data['Id_cl'] = 'SG';
        $data['ctr'] = $filepath;
        $data['nom_ctr'] = File::name($filepath);
        $data['Code_ctr'] = 'Zz01';

        return $this->sendRequest(
            $this->url('/v1/Arnmpsign2'),
            'POST',
            $certificat->cle_securite,
            $data
        ); 
    }

    public function getSignatureCount(Certificat $certificat): int
    {
        $response = $this->sendRequest(
            $this->url('/v1/getInfoCertificat/arnmp@dkbsignapi'),
            'GET',
            $certificat->cle_securite
        ); 

        return $this->parseSignatureCountResponse($response['nbre_doc_sign']);
    }

    public function getSignedDocumentItem(string $document): string
    {
        return $this->url('/include/DOCSIGN_ARNMP/' . $document);
    }

    //https://stackoverflow.com/a/15153931
    protected function base64ToImage(string $image): string
    {
        $filename = Str::random();

        $file = fopen(storage_path('app/public') . '/' . $filename, 'wb');
        $data = explode(',', $image);

        fwrite($file, base64_decode($data[1]));
        fclose($file);

        return $filename;
    }

    protected function parseSignatureCountResponse(string $response): int
    {
        $array = explode(' ', $response);
        return (int) $array[count($array) - 2];
    }
}
